from dataclasses import asdict, dataclass
from typing import Any, DefaultDict, Dict, List, Optional, Tuple

from pyrallis import field


@dataclass
class RTGTrainConfig:
    # wandb params
    project: str = "OSRL-baselines"
    group: str = None
    name: Optional[str] = None
    prefix: Optional[str] = "RTG_model"
    suffix: Optional[str] = ""
    logdir: Optional[str] = "logs"
    verbose: bool = True
    # dataset params
    outliers_percent: float = None
    noise_scale: float = None
    inpaint_ranges: Tuple[Tuple[float, float], ...] = None
    epsilon: float = None
    density: float = 1.0
    # model params
    embedding_dim: int = 128
    prompt_dim: int = 16
    state_encode_dim: int = 64
    state_encoder_hidden_sizes: List[float] = field(default=[128, 128, 128], is_mutable=True)
    r_hidden_sizes: List[float] = field(default=[256, 128, 128], is_mutable=True)
    linear_only: bool = False
    use_state: bool = True
    use_prompt: bool = True
    logprob_loss: bool = True
    need_rescale: bool = False
    # training params
    # task: str = "OfflineCarCircle-v0"
    task: str = "OfflinePointGoal1Gymnasium-v0"
    # context_encoder_path: str = "logs/context_encoder/ContextEncoder_decay_rate1.0-ba5c/ContextEncoder_decay_rate1.0-ba5c"
    context_encoder_path: str = "logs/context_encoder/ContextEncoder_simple_mlpTrue-d9c1/ContextEncoder_simple_mlpTrue-d9c1"
    dataset: str = None
    learning_rate: float = 1e-4
    batch_size: int = 2048
    epoch_num: int = 100
    steps_per_epoch: int = 1000
    reward_scale: float = 0.1
    cost_scale: float = 1
    num_workers: int = 8
    # evaluation params
    cost_limit: int = 10
    # general params
    seed: int = 0
    device: str = "cuda:1"
    threads: int = 6
    # augmentation param
    deg: int = 4
    pf_sample: bool = False
    beta: float = 1.0
    augment_percent: float = 0.2
    cost_reverse: bool = False
    # maximum absolute value of reward for the augmented trajs
    max_reward: float = 600.0
    # minimum reward above the PF curve
    min_reward: float = 1.0
    # the max drecrease of ret between the associated traj
    # w.r.t the nearest pf traj
    max_rew_decrease: float = 100.0
    # pf only mode param
    pf_only: bool = False
    rmin: float = 300
    cost_bins: int = 60
    npb: int = 5
    cost_sample: bool = False
    linear: bool = True  # linear or inverse
    start_sampling: bool = False
    prob: float = 0.2
    stochastic: bool = True
    init_temperature: float = 0.1
    no_entropy: bool = False
    # random augmentation
    random_aug: float = 0
    aug_rmin: float = 400
    aug_rmax: float = 500
    aug_cmin: float = -2
    aug_cmax: float = 25
    cgap: float = 5
    rstd: float = 1
    cstd: float = 0.2


@dataclass
class RTGCarCircleConfig(RTGTrainConfig):
    pass


@dataclass
class RTGAntRunConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineAntRun-v0"
    # augmentation param
    deg: int = 3
    max_reward: float = 1000.0
    max_rew_decrease: float = 150
    device: str = "cuda:2"


@dataclass
class RTGDroneRunConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 200
    # training params
    task: str = "OfflineDroneRun-v0"
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((400.0, 10), (500.0, 20), (600.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 700.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGDroneCircleConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineDroneCircle-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 1000.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGCarRunConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineCarRun-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 600.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGAntCircleConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineAntCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGBallRunConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineBallRun-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1400.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGBallCircleConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineBallCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1000.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class RTGCarButton1Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class RTGCarButton2Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class RTGCarCircle1Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class RTGCarCircle2Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class RTGCarGoal1Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 50.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class RTGCarGoal2Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class RTGCarPush1Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class RTGCarPush2Config(RTGTrainConfig):
    # training params
    task: str = "OfflineCarPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class RTGPointButton1Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    state_encoder_path: str = "logs/OfflinePointButton1Gymnasium-v0-cost-10/sa_encoder-660a/sa_encoder-660a_state_AE"
    # state_encoder_path: str = "logs/OfflinePointButton1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-894b/sa_encoder_idm_loss_weight0.0-894b_state_AE"


@dataclass
class RTGPointButton2Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointButton2Gymnasium-v0-cost-10/sa_encoder-e7e4/sa_encoder-e7e4_state_AE"
    state_encoder_path: str = "logs/OfflinePointButton2Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-b874/sa_encoder_idm_loss_weight0.0-b874_state_AE"


@dataclass
class RTGPointCircle1Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 65.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointCircle1Gymnasium-v0-cost-10/sa_encoder-2db5/sa_encoder-2db5_state_AE"
    state_encoder_path: str = "logs/OfflinePointCircle1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-2c5d/sa_encoder_idm_loss_weight0.0-2c5d_state_AE"


@dataclass
class RTGPointCircle2Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 55.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointCircle2Gymnasium-v0-cost-10/sa_encoder-1dbc/sa_encoder-1dbc_state_AE"
    state_encoder_path: str = "logs/OfflinePointCircle2Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-f4f4/sa_encoder_idm_loss_weight0.0-f4f4_state_AE"


@dataclass
class RTGPointGoal1Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:5"
    # state_encoder_path: str = "logs/OfflinePointGoal1Gymnasium-v0-cost-10/sa_encoder-82ac/sa_encoder-82ac_state_AE"
    state_encoder_path: str = "logs/OfflinePointGoal1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-62a4/sa_encoder_idm_loss_weight0.0-62a4_state_AE"


@dataclass
class RTGPointGoal2Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"


@dataclass
class RTGPointPush1Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGPointPush2Config(RTGTrainConfig):
    # training params
    task: str = "OfflinePointPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGAntVelocityConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:1"
    need_rescale: bool = True

@dataclass
class RTGAntVelocityV0Config(RTGTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:1"
    need_rescale: bool = True


@dataclass
class RTGHalfCheetahVelocityConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class RTGHalfCheetahVelocityV0Config(RTGTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class RTGHopperVelocityConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class RTGHopperVelocityV0Config(RTGTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class RTGSwimmerVelocityConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class RTGSwimmerVelocityV0Config(RTGTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class RTGWalker2dVelocityConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class RTGWalker2dVelocityV0Config(RTGTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class RTGEasySparseConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineMetadrive-easysparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGEasyMeanConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easymean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGEasyDenseConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easydense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGMediumSparseConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediumsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class RTGMediumMeanConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediummean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGMediumDenseConfig(RTGTrainConfig):
    # training params
    task: str = "OfflineMetadrive-mediumdense-v0"
    episode_len: int = 1000
    update_steps: int = 200_000


@dataclass
class RTGHardSparseConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGHardMeanConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardmean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class RTGHardDenseConfig(RTGTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-harddense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


RTG_DEFAULT_CONFIG = {
    # bullet_safety_gym
    "OfflineCarCircle-v0": RTGCarCircleConfig,
    "OfflineAntRun-v0": RTGAntRunConfig,
    "OfflineDroneRun-v0": RTGDroneRunConfig,
    "OfflineDroneCircle-v0": RTGDroneCircleConfig,
    "OfflineCarRun-v0": RTGCarRunConfig,
    "OfflineAntCircle-v0": RTGAntCircleConfig,
    "OfflineBallCircle-v0": RTGBallCircleConfig,
    "OfflineBallRun-v0": RTGBallRunConfig,
    # safety_gymnasium
    "OfflineCarButton1Gymnasium-v0": RTGCarButton1Config,
    "OfflineCarButton2Gymnasium-v0": RTGCarButton2Config,
    "OfflineCarCircle1Gymnasium-v0": RTGCarCircle1Config,
    "OfflineCarCircle2Gymnasium-v0": RTGCarCircle2Config,
    "OfflineCarGoal1Gymnasium-v0": RTGCarGoal1Config,
    "OfflineCarGoal2Gymnasium-v0": RTGCarGoal2Config,
    "OfflineCarPush1Gymnasium-v0": RTGCarPush1Config,
    "OfflineCarPush2Gymnasium-v0": RTGCarPush2Config,
    # safety_gymnasium: point
    "OfflinePointButton1Gymnasium-v0": RTGPointButton1Config,
    "OfflinePointButton2Gymnasium-v0": RTGPointButton2Config,
    "OfflinePointCircle1Gymnasium-v0": RTGPointCircle1Config,
    "OfflinePointCircle2Gymnasium-v0": RTGPointCircle2Config,
    "OfflinePointGoal1Gymnasium-v0": RTGPointGoal1Config,
    "OfflinePointGoal2Gymnasium-v0": RTGPointGoal2Config,
    "OfflinePointPush1Gymnasium-v0": RTGPointPush1Config,
    "OfflinePointPush2Gymnasium-v0": RTGPointPush2Config,
    # safety_gymnasium: velocity
    "OfflineAntVelocityGymnasium-v1": RTGAntVelocityConfig,
    "OfflineHalfCheetahVelocityGymnasium-v1": RTGHalfCheetahVelocityConfig,
    "OfflineHopperVelocityGymnasium-v1": RTGHopperVelocityConfig,
    "OfflineSwimmerVelocityGymnasium-v1": RTGSwimmerVelocityConfig,
    "OfflineWalker2dVelocityGymnasium-v1": RTGWalker2dVelocityConfig,
    "OfflineAntVelocityGymnasium-v0": RTGAntVelocityV0Config,
    "OfflineHalfCheetahVelocityGymnasium-v0": RTGHalfCheetahVelocityV0Config,
    "OfflineHopperVelocityGymnasium-v0": RTGHopperVelocityV0Config,
    "OfflineSwimmerVelocityGymnasium-v0": RTGSwimmerVelocityV0Config,
    "OfflineWalker2dVelocityGymnasium-v0": RTGWalker2dVelocityV0Config,
    # safe_metadrive
    "OfflineMetadrive-easysparse-v0": RTGEasySparseConfig,
    "OfflineMetadrive-easymean-v0": RTGEasyMeanConfig,
    "OfflineMetadrive-easydense-v0": RTGEasyDenseConfig,
    "OfflineMetadrive-mediumsparse-v0": RTGMediumSparseConfig,
    "OfflineMetadrive-mediummean-v0": RTGMediumMeanConfig,
    "OfflineMetadrive-mediumdense-v0": RTGMediumDenseConfig,
    "OfflineMetadrive-hardsparse-v0": RTGHardSparseConfig,
    "OfflineMetadrive-hardmean-v0": RTGHardMeanConfig,
    "OfflineMetadrive-harddense-v0": RTGHardDenseConfig
}
